<?php

namespace App\Models\api\v1;

use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Model;

/**
 * @property integer $id
 * @property integer $server_id
 * @property string $code
 * @property float $amount_due
 * @property float $rebate
 * @property float $remaining_amount
 * @property integer $order_status_id
 * @property boolean $is_active
 * @property string $created_at
 * @property integer $created_by
 * @property integer $updated_by
 * @property string $updated_at
 * @property boolean $is_deleted
 * @property string $deleted_at
 * @property integer $deleted_by
 * @property User $user
 * @property LineItem[] $lineItems
 * @property Transaction[] $transactions
 */
class Order extends Model
{
    /**
     * The "type" of the auto-incrementing ID.
     * 
     * @var string
     */
    protected $keyType = 'integer';

    /**
     * @var array
     */
    protected $fillable = [
        'server_id',
        'uuid',
        'amount_due',
        'customer_space_id',
        'rebate',
        'remaining_amount',
        'order_status_id',
        'order_number',
        'fees',
        'is_active',
        'day',
        'slots',
        'payment_method',
        'created_at',
        'created_by',
        'updated_by',
        'updated_at',
        'is_deleted',
        'deleted_at',
        'deleted_by',
        'table_number',
        'space_id'
    ];
    protected $hidden = ['created_by', 'updated_by', 'updated_at', 'is_deleted', 'deleted_at', 'deleted_by'];


    public static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            $model->uuid = Str::uuid();
            $model->created_by = auth()->user()->id ??  null;
            $model->updated_by = auth()->user()->id ??  null;
            $model->created_at = now();
            // $model->reference = Str::random(12);
        });

        static::updating(function ($model) {
            $model->updated_by = auth()->user()->id ??  null;
            $model->updated_at = now();
            // $model->reference = Str::random(12);
        });
    }
    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user()
    {
        return $this->belongsTo('App\Models\User', 'server_id');
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function customer()
    {
        return $this->belongsTo('App\Models\User', 'created_by');
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function space()
    {
        return $this->belongsTo('App\Models\api\v1\Space', 'space_id');
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function lineItems()
    {
        return $this->hasMany('App\Models\api\v1\LineItem');
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function transactions()
    {
        return $this->hasMany('App\Models\api\v1\Transaction');
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function orderStatus()
    {
        return $this->belongsTo(OrderStatus::class);
    }

    public static function getOrdersForSpace(int $space_id, int $step_number = 1): array
    {
        return Order::select('orders.*') //
            ->join('order_statuses', 'order_statuses.id', '=', 'orders.order_status_id')
            ->join('line_items', 'line_items.order_id', '=', 'orders.id')
            ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
            ->with(['orderStatus', 'lineItems.stock.item'])
            ->where([['orders.is_deleted', 0], ['orders.is_active', 1]])
            ->where([['stocks.space_id', $space_id]])
            ->where([['order_statuses.step_number', '>', $step_number]])
            ->orderBy('orders.created_at', 'asc')
            ->limit(20)
            ->distinct()
            ->get()
            ->toArray();
    }

    public static function getOrdersForCustomerSpace(int $space_id, int $wholesaler_id, String $status = null): array
    {
        $statusId = null;
        if (!empty($status)) {
            $statusId = OrderStatus::where('name', $status)->first()?->id;
        }
        $orders =  Order::select('orders.*') //
            ->join('order_statuses', 'order_statuses.id', '=', 'orders.order_status_id')
            ->join('line_items', 'line_items.order_id', '=', 'orders.id')
            ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
            ->with(['orderStatus', 'lineItems.stock.item'])
            ->where([['orders.is_deleted', 0], ['orders.is_active', 1]])
            ->where([['stocks.space_id', $wholesaler_id]])
            ->where('orders.customer_space_id', $space_id);
        if (!empty($statusId)) {
            $orders = $orders->where('orders.order_status_id', $statusId);
        }
        $orders = $orders->orderBy('orders.created_at', 'asc')
            ->limit(20)
            ->distinct()
            ->get()
            ->toArray();
        return $orders;
    }

    public static function getOrdersForSpaceByStatus(int $space_id, int $order_status_id, array $query = []): array
    {
        $orders =  Order::select('orders.*') //
            ->join('order_statuses', 'order_statuses.id', '=', 'orders.order_status_id')
            ->join('line_items', 'line_items.order_id', '=', 'orders.id')
            ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
            ->with(['orderStatus', 'lineItems.stock.item'])
            ->where([['orders.is_deleted', 0], ['orders.is_active', 1]])
            ->where([['stocks.space_id', $space_id]])
            ->where([['orders.order_status_id', $order_status_id]])
            ->whereNull('orders.server_id');
        if (!empty($query)) {
            $orders =  $orders->where($query);
        }
        $orders = $orders->orderBy('orders.created_at', 'asc')
            ->distinct()
            ->limit(20)
            ->get()
            ->toArray();
        return $orders;
    }

    public static function getHistoryOrdersForSpace(int $space_id, array $order_status_id, $limit = null, array $query = []): array
    {
        $orders = Order::select('orders.*') //
            ->join('order_statuses', 'order_statuses.id', '=', 'orders.order_status_id')
            ->join('line_items', 'line_items.order_id', '=', 'orders.id')
            ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
            ->with(['orderStatus', 'lineItems.stock.item'])
            ->where([['orders.is_deleted', 0], ['orders.is_active', 1]])
            ->where([['stocks.space_id', $space_id]])
            ->whereIn('orders.order_status_id', $order_status_id);
        if (!empty($query)) {
            $orders =  $orders->where($query);
        }
        if ($limit) {
            $orders = $orders->limit($limit);
        }
        return $orders->orderBy('orders.id', 'desc')->distinct()->get()
            ->toArray();
    }

    public static function getOrdersForSpaceToDeliver(int $space_id, int $order_status_id, int $serverid = null, array $query = []): array
    {
        $orders = Order::select('orders.*') //
            ->join('order_statuses', 'order_statuses.id', '=', 'orders.order_status_id')
            ->join('line_items', 'line_items.order_id', '=', 'orders.id')
            ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
            ->with(['orderStatus', 'lineItems.stock.item'])
            ->where([['orders.is_deleted', 0], ['orders.is_active', 1]])
            ->where([['stocks.space_id', $space_id]])
            ->where([['orders.order_status_id', $order_status_id]])
            ->whereNotNull('orders.server_id');
            if (!empty($serverid)) {
                $orders =  $orders->where('orders.server_id', $serverid);
            }
            if (!empty($query)) {
            $orders =  $orders->where($query);
        }
        $orders = $orders->orderBy('orders.created_at', 'asc')
            ->distinct()
            ->limit(20)
            ->get()
            ->toArray();
        return $orders;
    }
}
